"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.EncryptedHierarchicalKey = exports.HierarchicalSymmetricVersion = exports.ActiveHierarchicalSymmetricVersion = void 0;
const material_management_1 = require("@aws-crypto/material-management");
const constants_1 = require("./constants");
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#activehierarchicalsymmetric
//= type=implication
//# A structure that MUST have one member,
//# the UTF8 Encoded value of the version of the branch key.
class ActiveHierarchicalSymmetricVersion {
    constructor(activeVersion) {
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
        //# If the `type` attribute is equal to `"branch:ACTIVE"`
        //# then the authenticated encryption context MUST have a `version` attribute
        //# and the version string is this value.
        (0, material_management_1.needs)(activeVersion.startsWith(constants_1.BRANCH_KEY_TYPE_PREFIX), 'Unexpected branch key type.');
        (0, material_management_1.readOnlyProperty)(this, 'version', activeVersion.substring(constants_1.BRANCH_KEY_TYPE_PREFIX.length));
        Object.freeze(this);
    }
}
exports.ActiveHierarchicalSymmetricVersion = ActiveHierarchicalSymmetricVersion;
(0, material_management_1.immutableClass)(ActiveHierarchicalSymmetricVersion);
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#hierarchicalsymmetric
//= type=implication
//# A structure that MUST have one member,
//# the UTF8 Encoded value of the version of the branch key.
class HierarchicalSymmetricVersion {
    constructor(type_field) {
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
        //# If the `type` attribute start with `"branch:version:"` then the version string MUST be equal to this value.
        (0, material_management_1.needs)(type_field.startsWith(constants_1.BRANCH_KEY_TYPE_PREFIX), 'Type does not start with `branch:version:`');
        (0, material_management_1.readOnlyProperty)(this, 'version', type_field.substring(constants_1.BRANCH_KEY_TYPE_PREFIX.length));
        Object.freeze(this);
    }
}
exports.HierarchicalSymmetricVersion = HierarchicalSymmetricVersion;
(0, material_management_1.immutableClass)(HierarchicalSymmetricVersion);
class EncryptedHierarchicalKey {
    constructor(encryptionContext, ciphertextBlob) {
        (0, material_management_1.readOnlyProperty)(this, 'branchKeyId', encryptionContext[constants_1.BRANCH_KEY_IDENTIFIER_FIELD]);
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
        //# The `type` attribute MUST either be equal to `"branch:ACTIVE"` or start with `"branch:version:"`.
        (0, material_management_1.needs)(encryptionContext[constants_1.TYPE_FIELD] == constants_1.BRANCH_KEY_ACTIVE_TYPE ||
            encryptionContext[constants_1.TYPE_FIELD].startsWith(constants_1.BRANCH_KEY_TYPE_PREFIX), 'Unexpected branch key type.');
        (0, material_management_1.readOnlyProperty)(this, 'type', encryptionContext[constants_1.TYPE_FIELD] == constants_1.BRANCH_KEY_ACTIVE_TYPE
            ? new ActiveHierarchicalSymmetricVersion(encryptionContext[constants_1.BRANCH_KEY_ACTIVE_VERSION_FIELD])
            : new HierarchicalSymmetricVersion(encryptionContext[constants_1.TYPE_FIELD]));
        (0, material_management_1.readOnlyProperty)(this, 'createTime', encryptionContext[constants_1.KEY_CREATE_TIME_FIELD]);
        (0, material_management_1.readOnlyProperty)(this, 'kmsArn', encryptionContext[constants_1.KMS_FIELD]);
        (0, material_management_1.readOnlyProperty)(this, 'encryptionContext', Object.freeze({ ...encryptionContext }));
        (0, material_management_1.readOnlyProperty)(this, 'ciphertextBlob', ciphertextBlob);
        Object.freeze(this);
    }
}
exports.EncryptedHierarchicalKey = EncryptedHierarchicalKey;
(0, material_management_1.immutableClass)(EncryptedHierarchicalKey);
// This is a limited release for JS only.
// The full Key Store operations are available
// in the AWS Cryptographic Material Providers library
// in various languages (Java, .Net, Python, Rust...)
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#interface
//= type=exception
//# - [WriteNewEncryptedBranchKey](#writenewencryptedbranchkey)
//# - [WriteNewEncryptedBranchKeyVersion](#writenewencryptedbranchkeyversion)
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#interface
//= type=exception
//# - [GetEncryptedBeaconKey](#getencryptedbeaconkey)
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#writenewencryptedbranchkey
//= type=exception
//# The WriteNewEncryptedBranchKey caller MUST provide:
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#writenewencryptedbranchkeyversion
//= type=exception
//# The WriteNewEncryptedBranchKeyVersion caller MUST provide:
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#getencryptedbeaconkey
//= type=exception
//# The GetEncryptedBeaconKey caller MUST provide the same inputs as the [GetBeaconKey](../branch-key-store.md#getbeaconkey) operation.
//# It MUST return an [EncryptedHierarchicalKey](#encryptedhierarchicalkey).
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHlwZXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvdHlwZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7OztBQUl0Qyx5RUFJd0M7QUFDeEMsMkNBVW9CO0FBMkJwQixtR0FBbUc7QUFDbkcsb0JBQW9CO0FBQ3BCLDBDQUEwQztBQUMxQyw0REFBNEQ7QUFDNUQsTUFBYSxrQ0FBa0M7SUFHN0MsWUFBWSxhQUFtQztRQUM3Qyw2SEFBNkg7UUFDN0gseURBQXlEO1FBQ3pELDZFQUE2RTtRQUM3RSx5Q0FBeUM7UUFDekMsSUFBQSwyQkFBSyxFQUNILGFBQWEsQ0FBQyxVQUFVLENBQUMsa0NBQXNCLENBQUMsRUFDaEQsNkJBQTZCLENBQzlCLENBQUE7UUFDRCxJQUFBLHNDQUFnQixFQUNkLElBQUksRUFDSixTQUFTLEVBQ1QsYUFBYSxDQUFDLFNBQVMsQ0FBQyxrQ0FBc0IsQ0FBQyxNQUFNLENBQUMsQ0FDdkQsQ0FBQTtRQUVELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUE7SUFDckIsQ0FBQztDQUNGO0FBcEJELGdGQW9CQztBQUNELElBQUEsb0NBQWMsRUFBQyxrQ0FBa0MsQ0FBQyxDQUFBO0FBRWxELDZGQUE2RjtBQUM3RixvQkFBb0I7QUFDcEIsMENBQTBDO0FBQzFDLDREQUE0RDtBQUM1RCxNQUFhLDRCQUE0QjtJQUd2QyxZQUFZLFVBQWdDO1FBQzFDLDZIQUE2SDtRQUM3SCwrR0FBK0c7UUFDL0csSUFBQSwyQkFBSyxFQUNILFVBQVUsQ0FBQyxVQUFVLENBQUMsa0NBQXNCLENBQUMsRUFDN0MsNENBQTRDLENBQzdDLENBQUE7UUFDRCxJQUFBLHNDQUFnQixFQUNkLElBQUksRUFDSixTQUFTLEVBQ1QsVUFBVSxDQUFDLFNBQVMsQ0FBQyxrQ0FBc0IsQ0FBQyxNQUFNLENBQUMsQ0FDcEQsQ0FBQTtRQUNELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUE7SUFDckIsQ0FBQztDQUNGO0FBakJELG9FQWlCQztBQUNELElBQUEsb0NBQWMsRUFBQyw0QkFBNEIsQ0FBQyxDQUFBO0FBYTVDLE1BQWEsd0JBQXdCO0lBbUJuQyxZQUNFLGlCQUEyRSxFQUMzRSxjQUEwQjtRQUUxQixJQUFBLHNDQUFnQixFQUNkLElBQUksRUFDSixhQUFhLEVBQ2IsaUJBQWlCLENBQUMsdUNBQTJCLENBQUMsQ0FDL0MsQ0FBQTtRQUVELDZIQUE2SDtRQUM3SCxxR0FBcUc7UUFDckcsSUFBQSwyQkFBSyxFQUNILGlCQUFpQixDQUFDLHNCQUFVLENBQUMsSUFBSSxrQ0FBc0I7WUFDckQsaUJBQWlCLENBQUMsc0JBQVUsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxrQ0FBc0IsQ0FBQyxFQUNsRSw2QkFBNkIsQ0FDOUIsQ0FBQTtRQUVELElBQUEsc0NBQWdCLEVBQ2QsSUFBSSxFQUNKLE1BQU0sRUFDTixpQkFBaUIsQ0FBQyxzQkFBVSxDQUFDLElBQUksa0NBQXNCO1lBQ3JELENBQUMsQ0FBQyxJQUFJLGtDQUFrQyxDQUNwQyxpQkFBaUIsQ0FBQywyQ0FBK0IsQ0FBQyxDQUNuRDtZQUNILENBQUMsQ0FBQyxJQUFJLDRCQUE0QixDQUFDLGlCQUFpQixDQUFDLHNCQUFVLENBQUMsQ0FBQyxDQUNwRSxDQUFBO1FBQ0QsSUFBQSxzQ0FBZ0IsRUFDZCxJQUFJLEVBQ0osWUFBWSxFQUNaLGlCQUFpQixDQUFDLGlDQUFxQixDQUFDLENBQ3pDLENBQUE7UUFDRCxJQUFBLHNDQUFnQixFQUFDLElBQUksRUFBRSxRQUFRLEVBQUUsaUJBQWlCLENBQUMscUJBQVMsQ0FBQyxDQUFDLENBQUE7UUFDOUQsSUFBQSxzQ0FBZ0IsRUFDZCxJQUFJLEVBQ0osbUJBQW1CLEVBQ25CLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFBRSxHQUFHLGlCQUFpQixFQUFFLENBQUMsQ0FDeEMsQ0FBQTtRQUNELElBQUEsc0NBQWdCLEVBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxDQUFBO1FBRXhELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUE7SUFDckIsQ0FBQztDQUNGO0FBN0RELDREQTZEQztBQUNELElBQUEsb0NBQWMsRUFBQyx3QkFBd0IsQ0FBQyxDQUFBO0FBc0d4Qyx5Q0FBeUM7QUFDekMsOENBQThDO0FBQzlDLHNEQUFzRDtBQUN0RCxxREFBcUQ7QUFFckQsaUZBQWlGO0FBQ2pGLGtCQUFrQjtBQUNsQiwrREFBK0Q7QUFDL0QsNkVBQTZFO0FBRTdFLGlGQUFpRjtBQUNqRixrQkFBa0I7QUFDbEIscURBQXFEO0FBRXJELGtHQUFrRztBQUNsRyxrQkFBa0I7QUFDbEIsdURBQXVEO0FBRXZELHlHQUF5RztBQUN6RyxrQkFBa0I7QUFDbEIsOERBQThEO0FBRTlELDZGQUE2RjtBQUM3RixrQkFBa0I7QUFDbEIsdUlBQXVJO0FBQ3ZJLDRFQUE0RSJ9