# -*- coding: utf-8 -*- #
# Copyright 2025 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""services mcp list command."""

import sys

from googlecloudsdk.api_lib.services import services_util
from googlecloudsdk.api_lib.services import serviceusage
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.services import common_flags


@base.UniverseCompatible
@base.ReleaseTracks(base.ReleaseTrack.ALPHA)
class ListAlpha(base.ListCommand):
  """List MCP services for a project, folder or organization.

  This command lists the MCP services that are enabled or available (with MCP
  endpoints) to be MCP enabled
  by a project, folder or organization.

  ## EXAMPLES

  To list the services the current project has enabled for MCP, run:

    $ {command} --enabled

  To list the services the current project can enable for MCP, run:

    $ {command} --available
  """

  @staticmethod
  def Args(parser):
    """Args is called by calliope to gather arguments for this command.

    Args:
      parser: An argparse parser that you can use to add arguments that go on
        the command line after this command. Positional arguments are allowed.
    """
    mode_group = parser.add_mutually_exclusive_group(required=False)

    mode_group.add_argument(
        '--enabled',
        action='store_true',
        help=(
            '(DEFAULT) Return the MCP services which the project, folder or'
            ' organization has enabled.'
        ),
    )

    mode_group.add_argument(
        '--available',
        action='store_true',
        help=(
            'Return the services available to the '
            'project, folder or organization to enable for MCP.'
        ),
    )

    common_flags.add_resource_args(parser)

    base.PAGE_SIZE_FLAG.SetDefault(parser, 1000)

    # Remove unneeded list-related flags from parser
    base.URI_FLAG.RemoveFromParser(parser)

    parser.display_info.AddFormat("""
        table(
            name:label=NAME:sort=1,
            mcp_endpoint:label=MCP_ENDPOINT
        )
      """)

  def Run(self, args):
    """Run 'services mcp list'.

    Args:
      args: argparse.Namespace, The arguments that this command was invoked
        with.

    Returns:
      The list of MCP services for the given project, folder or organization.
    """
    # Default mode is --enabled, so if no flags were specified,
    # turn on the args.enabled flag.
    if not (args.enabled or args.available):
      args.enabled = True
    if args.IsSpecified('project'):
      project = args.project
    else:
      project = services_util.GetValidatedProject(args.project)
    if args.IsSpecified('folder'):
      folder = args.folder
    else:
      folder = None
    if args.IsSpecified('organization'):
      organization = args.organization
    else:
      organization = None

    if args.IsSpecified('limit'):
      limit = args.limit
    else:
      limit = sys.maxsize

    return serviceusage.ListMcpServicesV2Beta(
        project,
        args.enabled,
        args.page_size,
        limit=limit,
        folder=folder,
        organization=organization,
    )


# TODO(b/321801975) make command public after preview.
@base.UniverseCompatible
@base.Hidden
@base.ReleaseTracks(base.ReleaseTrack.BETA)
class List(base.ListCommand):
  """List MCP services for a project, folder or organization.

  This command lists the MCP services that are enabled or available (with MCP
  endpoints) to be MCP enabled
  by a project, folder or organization.

  ## EXAMPLES

  To list the services the current project has enabled for MCP, run:

    $ {command} --enabled

  To list the services the current project can enable for MCP, run:

    $ {command} --available
  """

  @staticmethod
  def Args(parser):
    """Args is called by calliope to gather arguments for this command.

    Args:
      parser: An argparse parser that you can use to add arguments that go on
        the command line after this command. Positional arguments are allowed.
    """
    mode_group = parser.add_mutually_exclusive_group(required=False)

    mode_group.add_argument(
        '--enabled',
        action='store_true',
        help=(
            '(DEFAULT) Return the MCP services which the project, folder or'
            ' organization has enabled.'
        ),
    )

    mode_group.add_argument(
        '--available',
        action='store_true',
        help=(
            'Return the services available to the '
            'project, folder or organization to enable for MCP.'
        ),
    )

    common_flags.add_resource_args(parser)

    base.PAGE_SIZE_FLAG.SetDefault(parser, 1000)

    # Remove unneeded list-related flags from parser
    base.URI_FLAG.RemoveFromParser(parser)

    parser.display_info.AddFormat("""
        table(
            name:label=NAME:sort=1,
            mcp_endpoint:label=MCP_ENDPOINT
        )
      """)

  def Run(self, args):
    """Run 'services mcp list'.

    Args:
      args: argparse.Namespace, The arguments that this command was invoked
        with.

    Returns:
      The list of MCP services for the given project, folder or organization.
    """
    # Default mode is --enabled, so if no flags were specified,
    # turn on the args.enabled flag.
    if not (args.enabled or args.available):
      args.enabled = True
    if args.IsSpecified('project'):
      project = args.project
    else:
      project = services_util.GetValidatedProject(args.project)
    if args.IsSpecified('folder'):
      folder = args.folder
    else:
      folder = None
    if args.IsSpecified('organization'):
      organization = args.organization
    else:
      organization = None

    if args.IsSpecified('limit'):
      limit = args.limit
    else:
      limit = sys.maxsize

    return serviceusage.ListMcpServicesV2Beta(
        project,
        args.enabled,
        args.page_size,
        limit=limit,
        folder=folder,
        organization=organization,
    )
