/*
 * Copyright (C) 2014-2021 Brian L. Browning
 *
 * This file is part of Beagle
 *
 * Beagle is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Beagle is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package vcf;

import beagleutil.ChromInterval;
import java.io.File;

/**
 * <p>Interface {@code GeneticMap} represents a genetic map for one or more
 * chromosomes.
 * </p>
 * <p>Instances of class {@code GeneticMap} are immutable.
 * </p>
 *
 * @author Brian L. Browning {@code <browning@uw.edu>}
 */
public interface GeneticMap {

    /**
     * Returns the base position corresponding to the specified genetic map
     * position. If the genetic position is not a map position then the base
     * position is estimated from the nearest genetic map positions using
     * linear interpolation.
     *
     * @param chrom the chromosome index
     * @param geneticPosition the genetic position on the chromosome
     * @return the base position corresponding to the specified genetic map
     * position
     * @throws IllegalArgumentException if the calculated base position
     * exceeds {@code Integer.MAX_VALUE}
     * @throws IllegalArgumentException if this genetic map has no
     * map positions for the specified chromosome
     * @throws IndexOutOfBoundsException if
     * {@code chrom < 0 || chrom >= ChromIds.instance().size()}
     */
    int basePos(int chrom, double geneticPosition);

    /**
     * Returns the genetic map position of the specified marker. The
     * genetic map position is estimated using linear interpolation.
     *
     * @param marker a genetic marker
     * @return the genetic map position of the specified marker
     * @throws IllegalArgumentException if this genetic map has no
     * map positions for the specified chromosome
     * @throws NullPointerException if {@code marker == null}
     */
    double genPos(Marker marker);

    /**
     * Returns the genetic map position of the specified genome coordinate.
     * The genetic map position is estimated using linear interpolation.
     *
     * @param chrom the chromosome index
     * @param basePosition the base coordinate on the chromosome
     * @return the genetic map position of the specified genome coordinate
     * @throws IllegalArgumentException if this genetic map has no
     * map positions for the specified chromosome
     * @throws IndexOutOfBoundsException if
     * {@code chrom < 0 || chrom >= ChromIds.instance().size()}
     */
    double genPos(int chrom, int basePosition);

    /**
     * Returns a string representation of this genetic map. The exact details
     * of the representation are unspecified and subject to change.
     *
     * @return a string representation of this genetic map
     */
    @Override
    String toString();

    /**
     * Constructs and returns a genetic map from the specified data.
     * If the specified map file is {@code null}, the returned genetic map
     * will convert genome coordinates to genetic units by dividing by
     * 1,000,000.  If {@code (chromInt != null)} the genetic map will
     * be restricted to chromosome {@code chromInt.chrom()}.
     * @param file a PLINK-format genetic map file with cM units
     * @param chromInt a chromosome interval
     * @return a genetic map from the specified data.
     * @throws IllegalArgumentException if any map position is infinite
     * or {@code NaN}
     * @throws NumberFormatException if the base position on any line of the map
     * file is not a parsable integer
     * @throws NumberFormatException if the genetic map position on any
     * line of the map file is not a parsable double
     * @throws IllegalArgumentException if a non-empty line of the specified
     * genetic map file does not contain 4 fields
     * @throws IllegalArgumentException if the map positions on each
     * chromosome are not sorted in ascending order
     * @throws IllegalArgumentException if there are duplicate
     * base positions on a chromosome
     * @throws IllegalArgumentException if all base positions on a chromosome
     * have the same genetic map position
     */
    static GeneticMap geneticMap(File file, ChromInterval chromInt) {
        if (file==null) {
            double scaleFactor = 1e-6;
            return new PositionMap(scaleFactor);
        }
        else {
            if (chromInt==null) {
                return PlinkGenMap.fromPlinkMapFile(file);
            }
            else {
                return PlinkGenMap.fromPlinkMapFile(file, chromInt.chrom());
            }
        }
    }

    /**
     * Returns the an array of length {@code markers.nMarkers()} whose
     * whose {@code j}-th element is the genetic map position of the
     * {@code j}-th marker.
     * @param genMap the genetic map
     * @param markers the list of markers
     * @return an array of genetic map positions
     * @throws IllegalArgumentException if
     * {@code markers.marker(0).chromIndex() != markers.marker(markers.nMarkers() - 1).chromIndex()}
     * @throws IllegalArgumentException if the specified genetic map has no
     * map positions for the specified chromosome
     * @throws NullPointerException if {@code genMap == null || markers == null}
     */
    static double[] genPos(GeneticMap genMap, Markers markers) {
        if (markers.marker(0).chromIndex()
                != markers.marker(markers.size()-1).chromIndex()) {
            throw new IllegalArgumentException("inconsistent data");
        }
        double[] genPos = new double[markers.size()];
        for (int j=0; j<genPos.length; ++j) {
            genPos[j] = genMap.genPos(markers.marker(j));
        }
        return genPos;
    }

    /**
     * Returns the an array of length {@code markers.nMarkers()} whose
     * whose {@code j}-th element is the genetic map position
     * of the {@code j}-th marker.
     * @param genMap the genetic map in cM units
     * @param minGenDist the required minimum cM distance between successive
     * markers
     * @param markers the list of markers
     * @return an array of genetic map positions
     * @throws IllegalArgumentException if
     * {@code markers.marker(0).chromIndex() != markers.marker(markers.nMarkers() - 1).chromIndex()}
     * @throws IllegalArgumentException if the specified genetic map has no
     * map positions for the specified chromosome
     * @throws IllegalArgumentException if {@code Double.isFinite(minDist) == false}
     * @throws NullPointerException if {@code genMap == null || markers == null}
     */
    static double[] genPos(GeneticMap genMap, double minGenDist, Markers markers) {
        if (markers.marker(0).chromIndex()
                != markers.marker(markers.size()-1).chromIndex()) {
            throw new IllegalArgumentException("inconsistent data");
        }
        if (Double.isFinite(minGenDist)==false) {
            throw new IllegalArgumentException(String.valueOf(minGenDist));
        }
        double[] genPos = new double[markers.size()];
        genPos[0] = genMap.genPos(markers.marker(0));
        double lastMapPos = genPos[0];
        for (int j=1; j<genPos.length; ++j) {
            double mapPos = genMap.genPos(markers.marker(j));
            double dist = Math.max((mapPos - lastMapPos), minGenDist);
            genPos[j] = genPos[j-1] + dist;
            lastMapPos = mapPos;
        }
        return genPos;
    }
}
