#!/usr/bin/env python3

import filecmp
import os
import re
import shutil
import subprocess
import tempfile

from setuptools import setup

VERSIONFILE = "px/version.py"


def write_version(version: str, replace: bool = False) -> None:
    """
    Write the given version to px/version.py.

    If px/version.py is missing, the version will always be written.

    If px/version.py is present, the version will only be written if replace is True.
    """
    with tempfile.NamedTemporaryFile(suffix=".py", delete=False) as tmp:
        tmp.write(
            b"# NOTE: Auto generated by update_version_py() in setup.py, no touchie!\n"
        )
        tmp.write(b'VERSION = "%s"\n' % bytearray(version, "utf_8"))

        # Flushing is required for filecmp.cmp() to work (below)
        tmp.flush()

        if not os.path.isfile(VERSIONFILE):
            # No version file found
            print(f"Creating {VERSIONFILE} with version {version}")
            shutil.move(tmp.name, VERSIONFILE)
        elif replace and not filecmp.cmp(tmp.name, VERSIONFILE):
            print(f"Updating {VERSIONFILE} to new version {version}")
            shutil.move(tmp.name, VERSIONFILE)
        else:
            # VERSIONFILE was already up to date. If we touch it in this case,
            # it will have its file timestamp updated, which will force the slow
            # px_integration_test.py tests to get rerun.
            #
            # Just clean up our tempfile and be merry.
            print(f"Not touching existing {VERSIONFILE}")
            os.remove(tmp.name)


def update_version_py() -> str:
    """
    Update px/version.py with the current git version.

    Returns the version number.
    """
    git_result = subprocess.run(["git", "describe", "--dirty"], capture_output=True)
    if git_result.returncode == 0:
        write_version(git_result.stdout.decode("utf-8").strip(), True)
    else:
        # Don't overwrite any existing version since we just made this one up
        write_version("0.0.0", False)

    from px import version

    return version.VERSION


if __name__ == "__main__":
    version_for_setuptools = update_version_py()

    if not re.match(r"^[0-9]+\.[0-9]+\.[0-9]+$", version_for_setuptools):
        # Setuptools requires a nice version number
        version_for_setuptools = "0.0.0"

    with open(
        os.path.join(os.path.dirname(__file__), "README.rst"), encoding="utf-8"
    ) as fp:
        LONG_DESCRIPTION = fp.read()

    setup(
        name="pxpx",
        version=version_for_setuptools,
        description="ps and top for Human Beings",
        long_description=LONG_DESCRIPTION,
        long_description_content_type="text/x-rst",
        author="Johan Walles",
        author_email="johan.walles@gmail.com",
        url="https://github.com/walles/px",
        license="MIT",
        classifiers=[
            "Development Status :: 5 - Production/Stable",
            "Environment :: Console",
            "Intended Audience :: System Administrators",
            "License :: OSI Approved :: MIT License",
            "Operating System :: MacOS",
            "Operating System :: POSIX :: Linux",
            "Programming Language :: Python :: 3",
            "Topic :: System :: Monitoring",
            "Topic :: System :: Systems Administration",
            "Topic :: Utilities",
        ],
        packages=["px"],
        # See: http://setuptools.readthedocs.io/en/latest/setuptools.html#setting-the-zip-safe-flag
        zip_safe=True,
        setup_requires=[
            "pytest-runner",
        ],
        entry_points={
            "console_scripts": [
                "px = px.px:main",
                "ptop = px.px:main",
                "pxtree = px.px:main",
            ],
        },
        # Note that we're by design *not* installing man pages here.
        # Using "data_files=" only puts the man pages in the egg file,
        # and installing that egg doesn't put them on the destination
        # system.
        #
        # After trying to figure this out for a bit, my conclusion is
        # that "pip install" simply isn't meant for installing any man
        # pages.
        #
        #   /johan.walles@gmail.com 2018aug27
    )
